Write-Host BIOSUpdater - Asus V1.0
<#
.SYNOPSIS
    ASUS BIOS Updater Script
.DESCRIPTION
    - Detects motherboard model via SMBIOS
    - Normalizes name to a safe folder name
    - Checks against board directory
    - Reads minimum required BIOS version from minversion.txt
    - Suspends BitLocker if enabled
    - Installs ASUS IO driver (AsIO3\InstDrv.exe -i)
    - Runs AsusFwUpdate.exe with the detected .CAP file
    - Logs actions to Logs\BIOSUpdate.log and results to Logs\BIOSUpdateResults.csv
    - Exit codes:
        0   = Success
        10  = Board not supported
        11  = No BIOS file found
        12  = BIOS version already up-to-date (skipped)
        20  = Missing or unreadable minversion.txt
        30  = Failed to install ASUS IO driver
        40+ = Flasher tool error
#>

# ----------------------------
# Setup paths
# ----------------------------
$ScriptRoot = Split-Path -Parent $MyInvocation.MyCommand.Definition
$LogDir     = Join-Path $ScriptRoot "Logs"
$ResultCsv  = Join-Path $LogDir "BIOSUpdateResults.csv"
$LogFile    = Join-Path $LogDir "BIOSUpdate.log"
$BoardsDir  = Join-Path $ScriptRoot "Boards"
$DriverDir  = Join-Path $ScriptRoot "AsIO3"
$DriverExe  = Join-Path $DriverDir "InstDrv.exe"
$FlashExe   = Join-Path $DriverDir "AsusFwUpdate.exe"

# Ensure Logs dir exists
if (-not (Test-Path $LogDir)) { New-Item -Path $LogDir -ItemType Directory -Force | Out-Null }

# ----------------------------
# Logging helpers
# ----------------------------
function Write-Log {
    param([string]$Message)
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $entry = "$timestamp - $Message"
    Add-Content -Path $LogFile -Value $entry
    Write-Host $entry
}

function Write-ResultRow {
    param(
        [string]$Board,
        [string]$Serial,
        [string]$CurrentVer,
        [string]$TargetVer,
        [string]$Status,
        [int]$ExitCode
    )

    if (-not (Test-Path $ResultCsv)) {
        "Timestamp,SerialNumber,Baseboard,CurrentVersion,TargetVersion,Status,ExitCode" | Out-File $ResultCsv -Encoding UTF8
    }
    $row = "{0},{1},{2},{3},{4},{5},{6}" -f (Get-Date -Format "yyyy-MM-dd HH:mm:ss"),$Serial,$Board,$CurrentVer,$TargetVer,$Status,$ExitCode
    Add-Content -Path $ResultCsv -Value $row
}

# ----------------------------
# Detect system info
# ----------------------------
$baseboardRaw = (Get-CimInstance Win32_BaseBoard).Product.Trim()
$serial       = (Get-CimInstance Win32_BIOS).SerialNumber.Trim()
$currentVer   = (Get-CimInstance Win32_BIOS).SMBIOSBIOSVersion.Trim()

Write-Log "Detected board (raw SMBIOS): $baseboardRaw"
Write-Log "Serial number: $serial"
Write-Log "Current BIOS version: $currentVer"

# Normalize SMBIOS name into filesystem-safe folder name
$boardFolderName = ($baseboardRaw -replace '[^A-Za-z0-9\-]', '-').Trim('-')
$boardDir = Join-Path $BoardsDir $boardFolderName

Write-Log "Looking for board folder: $boardFolderName"

if (-not (Test-Path $boardDir)) {
    Write-Log "ERROR: No matching board directory for $boardFolderName"
    Write-ResultRow $baseboardRaw $serial $currentVer "N/A" "Board not supported" 10
    exit 10
}

# ----------------------------
# Read minimum required version
# ----------------------------
$minVerFile = Join-Path $boardDir "minversion.txt"
if (-not (Test-Path $minVerFile)) {
    Write-Log "ERROR: Missing minversion.txt in $boardDir"
    Write-ResultRow $baseboardRaw $serial $currentVer "N/A" "Missing minversion.txt" 20
    exit 20
}
$targetVer = Get-Content $minVerFile | Select-Object -First 1
Write-Log "Minimum required BIOS version: $targetVer"

if ($currentVer -ge $targetVer) {
    Write-Log "BIOS is already at or above required version ($currentVer >= $targetVer). Skipping."
    Write-ResultRow $baseboardRaw $serial $currentVer $targetVer "Skipped (Up-to-date)" 12
    exit 12
}

# ----------------------------
# Find BIOS .CAP file
# ----------------------------
$capFile = Get-ChildItem -Path $boardDir -Filter *.CAP | Select-Object -First 1
if (-not $capFile) {
    Write-Log "ERROR: No .CAP file found in $boardDir"
    Write-ResultRow $baseboardRaw $serial $currentVer $targetVer "No BIOS file found" 11
    exit 11
}
$capPath = $capFile.FullName
Write-Log "Using BIOS file: $capPath"

# ----------------------------
# Suspend BitLocker if enabled
# ----------------------------
$blv = Get-BitLockerVolume -ErrorAction SilentlyContinue
if ($blv) {
    $protectors = $blv | Where-Object { $_.ProtectionStatus -eq "On" }
    if ($protectors) {
        Write-Log "Suspending BitLocker protection..."
        Suspend-BitLocker -MountPoint $protectors.MountPoint -RebootCount 1
    }
}

# ----------------------------
# Install ASUS IO driver
# ----------------------------
if (-not (Test-Path $DriverExe)) {
    Write-Log "ERROR: Missing ASUS driver installer at $DriverExe"
    Write-ResultRow $baseboardRaw $serial $currentVer $targetVer "Missing driver installer" 30
    exit 30
}

Write-Log "Installing ASUS IO driver..."
$drv = Start-Process -FilePath $DriverExe -ArgumentList "-i" -Wait -PassThru
if ($drv.ExitCode -ne 0) {
    Write-Log "ERROR: Driver installation failed with exit code $($drv.ExitCode)"
    Write-ResultRow $baseboardRaw $serial $currentVer $targetVer "Driver install failed" 31
    exit 31
}

# ----------------------------
# Run BIOS flasher
# ----------------------------
if (-not (Test-Path $FlashExe)) {
    Write-Log "ERROR: Missing AsusFwUpdate.exe in $DriverDir"
    Write-ResultRow $baseboardRaw $serial $currentVer $targetVer "Missing flasher" 40
    exit 40
}

$flashArgs = "-FlashBios /P /U `"$capPath`""
Write-Log "Running flasher: AsusFwUpdate.exe $flashArgs"

$proc = Start-Process -FilePath $FlashExe -ArgumentList $flashArgs -Wait -PassThru
$exitCode = $proc.ExitCode

if ($exitCode -eq 0) {
    Write-Log "BIOS update completed successfully"
    Write-ResultRow $baseboardRaw $serial $currentVer $targetVer "Success" 0
    Write-Host "BIOS update completed successfully. Please reboot."
    exit 0
} else {
    Write-Log "ERROR: Flasher returned exit code $exitCode"
    Write-ResultRow $baseboardRaw $serial $currentVer $targetVer "Error: Flasher failed" $exitCode
    Write-Host "BIOS update failed with exit code $exitCode. Check logs for details."
    exit $exitCode
}
